--[[----------------------------------------------------------------------------
	Copyright 2008 João Liborio Cardoso
	All rights reserved
	
	Bongos Durability is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License version 3 as published by the Free Software Foundation.
	Bongos Durability is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License (<http://www.gnu.org/licenses/>) for more details.
	Every lua file included in this addon is part of Bongos Durability and is subject to the same terms as the ones described here.
---------------------------------------------------------------------------]]--

local Bongos = LibStub('AceAddon-3.0'):GetAddon('Bongos3')
local Durability = Bongos:NewModule('Durability', 'Collision-1.3')
local AutoBongos = LibStub('AutoBongos-1.0')

local L = LibStub('AceLocale-3.0'):GetLocale('Bongos3-Durability')
local new, old


--[[ Startup ]]--

function Durability:Load()
	local defaults = {
		point = "TOPRIGHT",
		damaged = {},
		broken = {},
		y = -431.1,
		x = -14,
	}

	local bar, isNew = AutoBongos:CreateBar('durability', defaults, nil, 'MEDIUM')
	if isNew then
		self.LoadBar(bar)
	end
	
	bar.IndicatorEvents = {"CHAT_MSG_COMBAT_MISC_INFO", "PLAYER_REGEN_ENABLED"}
	bar:ToggleIndicator(bar.sets.showPercentage)
	
	bar:UpdateText()
	bar:OnUpdate()
	
	self.bar = bar
end

function Durability:Unload()
	self.bar:Destroy()
end

function Durability:LoadBar()
	for k,v in pairs(Durability) do self[k] = v end
	
	local frame = DurabilityFrame
	self:EnableMouse(true)
	self:Attach(frame)
	self.frame = frame
	
	self.indicator = self:CreateIndicator()
	self:SetScript('OnEnter', self.OnEnter)
	self:SetScript('OnLeave', self.OnLeave)
	self:RegisterEvent("UPDATE_INVENTORY_ALERTS", "OnUpdate")
end


--[[ Updates ]]--

function Durability:OnUpdate()
	self:UpdateSize()
	if self.sets.warnings then
		self:PrintWarnings("Damaged", 1, 0.82, 0)
		self:PrintWarnings("Broken", 1, 0.1, 0.1)
	end
end

function Durability:UpdateSize()
	if self.frame then	
		local off, width, height = 0
		
		if Bongos:IsLocked() and not self.frame:IsShown() then
			width, height = 0.1, 0.1
		else
			width, height = 80, 83
			if DurabilityShield:IsShown() or DurabilityOffWeapon:IsShown() or DurabilityRanged:IsShown() then
				width = width + 10
				off = off - 5
			end
			if DurabilityWeapon:IsShown() then
				width = width + 10
				off = off + 5
			end
		end
		
		self.ClearAllPoints(self.frame)
		self.SetPoint(self.frame, 'TOP', self, 'TOP', off, -3)
		
		self:SetHeight(height)
		self:SetWidth(width)
	end
end


--[[ Percentage Indicator ]]--

function Durability:CreateIndicator()
	local indicator = self.frame:CreateFontString("DurabilityFrameBongosIndicator")
	indicator:SetFontObject("GameFontNormal")
	indicator:SetPoint("Center", DurabilityFrame, "Center", 2.5, 2)
	indicator:SetJustifyH("CENTER")
	indicator:Show()
	
	return indicator
end

function Durability:ToggleIndicator(enable)
	self.sets.showPercentage = enable
	if enable then
		self:OnEnter()
		self:EnableMouse(false)
		for _,event in pairs(self.IndicatorEvents) do
			self:RegisterEvent(event, "UpdateText")
		end
	else
		self:OnLeave()
		self:EnableMouse(true)
		for i,event in pairs(self.IndicatorEvents) do
			self:UnregisterEvent(event)
		end
	end
end

function Durability:UpdateText()
	local value = self:GetOverallDurability()
	local r,g,b
	
	if value > 65 then
		r, g, b = 0.1, 1, 0.1
	elseif value > 25 then
		r, g, b = 1, 0.82, 0
	else
		r, g, b = 1, 0.1, 0.1
	end
	
	self.indicator:SetText(value.."%")
	self.indicator:SetTextColor(r, g, b)
end

function Durability:OnEnter()
	self:UpdateText()
	UIFrameFadeIn(self.indicator, 0.3)
end

function Durability:OnLeave()
	UIFrameFadeOut(self.indicator, 0.3)
end


--[[ Warnings ]]--

function Durability:PrintWarnings(string,...)
	local iterator = self["Iterate"..string.."Armor"]
	old = self.sets[strlower(string)]
	new = {}
	
	for i, slot in iterator() do
		if not old[slot] then
			UIErrorsFrame:AddMessage(format(L[string], rawget(L, slot) or slot), ...)
		end
		new[slot] = true
	end
	
	self.sets[strlower(string)] = new
end


--[[ Menu ]]--

function Durability:CreateMenu()

	local menu = Bongos.Menu:Create(self.id)
	local panel = menu:AddLayoutPanel()
	
	local showPercentage = panel:CreateCheckButton(L.ShowPercentage)
	showPercentage:SetScript('OnShow', function() this:SetChecked(self.sets.showPercentage) end)
	showPercentage:SetScript('OnClick', function() self:ToggleIndicator(this:GetChecked()) end)
	
	local warnings = panel:CreateCheckButton(L.EnableWarnings)
	warnings:SetScript('OnShow', function() this:SetChecked(self.sets.warnings) end)
	warnings:SetScript('OnClick', function() self.sets.warnings = this:GetChecked() end)
	
	return menu
	
end